
// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Emboss.3x3.1.fsh                               ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.5 (2017/02/13)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To emboss an image.                                           ==
// ==                                                                        ==
// == Description: A 3x3 filter is used to create an embossing effect  on  a ==
// == source image. The filter's sample point spacing can be specified via a ==
// == control parameter input to the shader as a uniform  variable  and  the ==
// == filter can optionaly be repeated with increasing sample spacings.      ==
// ==                                                                        ==
// == The filter is a simple one,  in  particular it only samples the source ==
// == at a small set of points and ignores other points in the neighbourhood ==
// == of each pixel, even those within the area being sampled for the pixel. ==
// == For small sample point spacings this works well but when larger sample ==
// == point spacings are used the result may exhibit "spurious" artefacts of ==
// == various kinds, however these often are of interrest in their own right ==
// == as they add texture to the image.                                      ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/01/22 - v1.0.0 - Initial version.                               ==
// ==   2017/01/23 - v1.0.1 - Added sample mode and sample size parameters.  ==
// ==   2017/01/23 - v1.0.2 - Simplified the filter definition mechanism.    ==
// ==   2017/01/23 - v1.0.3 - Added an edge wrap around option.              ==
// ==   2017/01/25 - v1.0.4 - Added angle, added aspect ratio equalisation.  ==
// ==   2017/02/13 - v1.0.5 - Corrected spelling for normalisation function. ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Filter definition =======================================================
// ============================================================================

// The shader is tailored to a particular filter via the macros defined  here.
// A filter is specified as a sum of contributions from each sample point. The
// QQ macro, defined in the body of the shader, is used to specify each sample
// point,  its arguments being the X and Y positions of the sample relative to
// the filter's center and the weight to be used for the sample.  The  weights
// do not have to be normalised as that is acheived  via  the  overall  WEIGHT
// factor which is either specified separately or,  more commonly,  calculated
// automatically as the sum of the individual weights.
//
// If the weight at a point is 0.0 then the point should preferably be omitted
// from the filter definition in order to reduce the processing power required
// for filter evaluation.
//
// The filter normalisation factor can be specified manualy by defining WEIGHT
// as a macro with the required value.  If  WEIGHT is nor defined then it will
// be automaticly defined as the sum of the individual sample point weights.
//
// Normally filter should be defined as a sum of QQ terms,  but other forms of
// expression can be used, however when a non-standard form is used for FILTER
// the automatic calculation of WEIGHT should not be used as it is only  valid
// for the standard form of filter expression.
//
// It is convenient,  but not necessary,  to lay out the filter definitions so
// that they reflect the sample point geometry.

#define FILTER QQ(-1.0,-1.0,-4.0)                                         \
                                  + QQ(+0.0,+0.0,5.0)                     \
                                                      + QQ(+1.0,+1.0,4.0) \

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

vec2 Emu_Normalise_to_Window ( vec2 xy ) { return xy / u_WindowSize.xy; }

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Boolean_Vector_Ops.lib.src

#define Emu_bvec2_or(A,B)  bvec2 ( A[0]||B[0], A[1]||B[1] )
#define Emu_bvec2_and(A,B) bvec2 ( A[0]&&B[0], A[1]&&B[1] )
#define Emu_bvec2_xor(A,B) bvec2 ( A[0]^^B[0], A[1]^^B[1] )

#define Emu_bvec2_or_not(A,B)  Emu_bvec2_or  ( A, not(B) )
#define Emu_bvec2_and_not(A,B) Emu_bvec2_and ( A, not(B) )
#define Emu_bvec2_xor_not(A,B) Emu_bvec2_xor ( A, not(B) )

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

uniform vec2 Emu_Emboss_RA_scale;
uniform vec2 Emu_Emboss_RA_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Emboss_RA_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Emboss_RA_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default any sample points lying outside the area  to  be  embossed  are
// replaced by points on its edge but this can be overridden by specifying an
// alternate edge mode is to be used.  The two components control the X and Y
// edge pairs, it is not possible to control each edge separately.  The valid 
// values for the edge modes are
//
//     0 - Edge overflows will smear. Default.
//     1 - Edge overflows fade to transparent.
//     2 - Edge overflows wrap round.
//
// Unfortunately iStripper does not support passing ivecs from scene files so
// a float vector has to be used for Emu_Emboss_edge_mode.

uniform vec2 Emu_Emboss_edge_mode;

ivec2 edge_mode  = ivec2 ( Emu_Emboss_edge_mode );
bvec2 edge_undef = greaterThan ( edge_mode, ivec2(2) );

bvec2 edge_wraps =                equal ( edge_mode, ivec2(2) ); 
bvec2 edge_fades =                equal ( edge_mode, ivec2(1) );
bvec2 edge_smear = Emu_bvec2_or ( equal ( edge_mode, ivec2(0) ), edge_undef );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the filter sample point spacing is the resolution of the current
// graphics window but this may be changed using the following parameters:
//
//    Emu_Emboss_sample_mode
//    Emu_Emboss_sample_size 
//    Emu_Emboss_sample_step
//
// The sample_mode parameter is a two decimal digit integer, ES, where each of
// the digits controls a different aspect of the shader. The least significant
// digit, S, selects between alternatives for the source image size and may be
//
//     0 - Use the default selection
//     1 - Use Emu_Emboss_sample_size 
//     2 - Use the graphics window size
//     3 - Use the image's texture size
//
// The more significant digit, E, controls aspect ratio  equalisation.  When no
// aspect ration equalisation is used then for a "square",  i.e. an NxN filter,
// the filter will act over a rectangular region with the same aspect ratio  as
// the image being filtered, or more acurately,  the sample_size ratio.  It  is
// sometimes desirable that area acted on by the filter is square,  e.g. when a
// "pixelate" shader is being used it is often desirable that the pixels appear
// square when the image is displayed.  The valid values for this parameter are
//
//     0 - No aspect ratio equalisation performed
//     1 - Use the Emu_Emboss_sample_size ratio
//     2 - Use the graphics window's aspect ratio
//     3 - Use the image's texture's aspect ratio
//
// The sample_size parameter specifies an explicit size for the  source  image.
// If it is not specified then either the graphics window size or  the  texture
// size will be selected depending on the sample_mode.
//
// By default the sample spacing is the reciprocal of the sample size, i.e. its
// nominal resolution which will be that of the window or of the texture unless
// overridden by an explicit sample size. However, this spacing can modified by
// scaling it by sample step factors which default to (1.0,1.0).
//
// If default values are used for the sample size and sample  step  the  filter
// will behave normaly,  but when values significantly different are used, e.g.
// a sample step of more than 4.0,  then filter artefacts may be produced which
// will affect the texture of the final image - often in interresting ways.

uniform int  Emu_Emboss_sample_mode;
uniform vec2 Emu_Emboss_sample_size;
uniform vec2 Emu_Emboss_sample_step;

// Isolate Size Selection and Aspect Ratio Size Equalisation control parts of
// the sample mode.  Sorry, but I was unable to resist the acronym "Arse" for
// Aspect Ratio Size Equalisation.

int sample_mode_SSel = int(mod(float(Emu_Emboss_sample_mode),10.0));
int sample_mode_Arse = int(mod(float(Emu_Emboss_sample_mode)/10.0,10.0));

// Select the sample size specified by sample_mode_SSel.

vec2 default_size = ( sample_mode_SSel == 2 ) ? textureSize(iChannel0,0)
                                              : u_WindowSize;

vec2 sample_sssss = EMU_DEFAULT ( vec2, Emu_Emboss_sample_size, default_size );
 
vec2 sample_size = default_size             * float ( sample_mode_SSel == 0 )
                 + sample_sssss             * float ( sample_mode_SSel == 1 )
                 + textureSize(iChannel0,0) * float ( sample_mode_SSel == 2 )
                 + u_WindowSize             * float ( sample_mode_SSel == 3 );

// Select which size to use for the aspect ratio equalisation.

vec2 arse_size = default_size             * float ( sample_mode_Arse == 0 )
               + sample_sssss             * float ( sample_mode_Arse == 1 )
               + textureSize(iChannel0,0) * float ( sample_mode_Arse == 2 )
               + u_WindowSize             * float ( sample_mode_Arse == 3 );

// Calculate the aspect ratio equalisation factor.

vec2 arse_ratio = ( sample_mode_Arse == 0 ) ? vec2(1.0)
                : min ( arse_size.yx/arse_size.xy, vec2(1.0) );

// Determine the sample step to use for the filter.

vec2 sample_step = EMU_DEFAULT ( vec2, Emu_Emboss_sample_step, vec2(1.0) )
                 * arse_ratio / sample_size;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The filter may be rotated by an arbitrary angle.

uniform float Emu_Blur_angle;

float angle = radians ( Emu_Blur_angle );

mat2 rotate = mat2 ( cos(angle), sin(angle),
                    -sin(angle), cos(angle)
                   );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The action of the filter may optionaly be repeated any number of times. A
// value of 0 will be replaced by the default value, 1, but a negative value
// can be used to suppress the filter.

uniform int Emu_Emboss_repeat_count;

int repeat_count = max ( 0, EMU_DEFAULT ( int, Emu_Emboss_repeat_count, 1 ) );

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point and apply
   // standard scaling and hotspot options if any were requested.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   uv = ( uv - hotspot ) / scale + hotspot;

   // Apply the requested rounds of filtering. The macros used when
   // defining the filter are themselves defined here because  they
   // reference some local variables and it is convenient  to  have
   // everything together rather than define the macros elsewhere.

   #define EX(xy)       Emu_bvec2_and ( edge_fades, not(equal(xy,fract(xy))) )
   #define CL(xy)       clamp ( xy, vec2(0.0), vec2(1.0) )
      
   #define ST(xy)       mix ( CL(xy), fract(xy), vec2(edge_wraps) )
   #define TX(xy,ww)    texture2D ( iChannel0, ST(xy*ss+uv) )

   #define CC(xy,ww)    mix ( TX(xy,ww)*(ww), vec4(0.0), vec4(any(EX(xy))) )
   #define QQ(xx,yy,ww) CC ( vec2(xx,yy), ww )

   vec4 colour = vec4(0.0);

   for ( int i=0; i<repeat_count; i++ )
    { vec2 ss = sample_step * (i+1);
      colour += FILTER;
   }

   // If necessary calculate the flter's normalisation factor as the
   // sum of the weights for the individual sample points.  This  is
   // done by redefining the QQ macro changing the effect of FILTER.

   #ifndef WEIGHT
      #undef  QQ
      #define QQ(xx,yy,ww) +(ww)
      #define WEIGHT (FILTER)
   #endif

   colour /= float(repeat_count) * (WEIGHT);

   // Update shader outputs.

   gl_FragColor = colour * gl_Color;

}